<?php
/**
 * CodeBrain BV       https://www.codebrain.nl.
 *
 * @author      CodeBrain B.V. <info@codebrain.nl>
 * @copyright   Codebrain B.V.
 * @license     https://opensource.org/licenses/bsd-license.php
 *
 * @see        https://www.ideal-checkout.nl/
 */
require _PS_MODULE_DIR_.'icomnikassa'.DIRECTORY_SEPARATOR.'libraries'.DIRECTORY_SEPARATOR.'omnikassa.cls.php';
require _PS_MODULE_DIR_.'icomnikassa'.DIRECTORY_SEPARATOR.'libraries'.DIRECTORY_SEPARATOR.'ic.functions.php';

use PrestaShop\PrestaShop\Core\Payment\PaymentOption;

if (!defined('_PS_VERSION_')) {
    exit;
}

class IcOmnikassa extends PaymentModule
{
    protected $sHtml = '';
    protected $aPostErrors = [];

    public $details;
    public $owner;
    public $address;
    public $extra_mail_vars;

    public $is_eu_compatible;
    public $aModuleConfig;
    public $aPaymentMethods;

    public function __construct()
    {
        $this->name = 'icomnikassa';
        $this->tab = 'payments_gateways';
        $this->version = '1.0.9';
        $this->author = 'iDEAL Checkout';
        $this->module_key = '3621c328f584d23528a907c7d3356b54';
        $this->ps_versions_compliancy = ['min' => '1.7', 'max' => _PS_VERSION_];
        $this->controllers = ['payment', 'validation'];
        $this->is_eu_compatible = 1;
        $this->need_instance = 0;

        $this->currencies = true;
        $this->currencies_mode = 'checkbox';

        $this->bootstrap = true;
        parent::__construct();

        $this->displayName = $this->l('iDEAL Checkout (Native) - Rabo Smart Pay');
        $this->description = $this->l('Accept online payments through Rabo Smart Pay');

        $this->confirmUninstall = $this->l('Are you sure you want to uninstall?');

        $this->aModuleConfig = [
            'ICROK2_WEBHOOK_URL' => '',
            'ICROK2_REFRESHTOKEN_PRODUCTION' => '',
            'ICROK2_SIGNINGKEY_PRODUCTION' => '',
            'ICROK2_REFRESHTOKEN_SANDBOX' => '',
            'ICROK2_SIGNINGKEY_SANDBOX' => '',
            'ICROK2_SANDBOX' => 0,
            'ICROK2_COF' => 0,
            'ICROK2_OS_CANCELED' => (int) Configuration::get('PS_OS_CANCELED'),
            'ICROK2_OS_PROCESSING' => (int) Configuration::get('PS_OS_PAYMENT'),
            'ICROK2_OS_FAILURE' => (int) Configuration::get('PS_OS_ERROR'),
            'ICROK2_OS_PENDING' => (int) Configuration::get('ICROK2_OS_AWAITING'),
        ];

        $this->aPaymentMethods = [];
        $this->aPaymentMethods[] = ['id' => 'ideal', 'name' => $this->l('iDEAL')];
        $this->aPaymentMethods[] = ['id' => 'bancontact', 'name' => $this->l('Bancontact')];
        $this->aPaymentMethods[] = ['id' => 'maestro', 'name' => $this->l('Maestro')];
        $this->aPaymentMethods[] = ['id' => 'mastercard', 'name' => $this->l('Mastercard')];
        $this->aPaymentMethods[] = ['id' => 'paypal', 'name' => $this->l('PayPal')];
        $this->aPaymentMethods[] = ['id' => 'v_pay', 'name' => $this->l('Vpay')];
        $this->aPaymentMethods[] = ['id' => 'visa', 'name' => $this->l('Visa')];
        $this->aPaymentMethods[] = ['id' => 'cards', 'name' => $this->l('Creditcards / Apple Pay')];

        for ($i = 0; $i < count($this->aPaymentMethods); ++$i) {
            $sConfig = Configuration::get('ICROK2_SORTORDER_'.$this->aPaymentMethods[$i]['id']);

            $this->aPaymentMethods[$i]['sortOrder'] = $sConfig;

            if (empty($this->aPaymentMethods[$i]['sortOrder'])) {
                $this->aPaymentMethods[$i]['sortOrder'] = 100 + $i;
            }
        }

        usort($this->aPaymentMethods, function ($a, $b) {
            return $a['sortOrder'] - $b['sortOrder'];
        });
    }

    public function install()
    {
        if ($this->installOrderState() == false) {
            return false;
        }

        $this->aModuleConfig['ICROK2_OS_PENDING'] = (int) Configuration::get('ICROK2_OS_PENDING');

        foreach ($this->aModuleConfig as $k => $v) {
            // Multistore Active?
            if (Shop::isFeatureActive()) {
                $aShops = Shop::getShops();

                foreach ($aShops as $aShop) {
                    // Update configuration for each store
                    if (!Configuration::updateValue($k, $v, false, null, (int) $aShop['id_shop'])) {
                        return false;
                    }
                }
            } else {
                // Update configuration for this store
                if (!Configuration::updateValue($k, $v)) {
                    return false;
                }
            }
        }

        if (!parent::install() || !$this->registerHook('paymentOptions') || !$this->registerHook('displayPaymentReturn')) {
            return false;
        }

        return true;
    }

    public function uninstall()
    {
        $aModuleConfigKeys = array_keys($this->aModuleConfig);

        foreach ($aModuleConfigKeys as $sModuleConfigKey) {
            if (!Configuration::deleteByName($sModuleConfigKey)) {
                return false;
            }
        }

        if (!parent::uninstall()) {
            return false;
        }

        if ($this->uninstallOrderStates() == false) {
            return false;
        }

        return true;
    }

    public function getContent()
    {
        $this->sHtml = '';

        if (Tools::isSubmit('submit_'.$this->name)) {
            $this->postValidation();

            if (!count($this->aPostErrors)) {
                $this->postProcess();
            } else {
                foreach ($this->aPostErrors as $sError) {
                    $this->sHtml .= $this->displayError($sError);
                }
            }
        }

        $aForm = [];

        // Generate the Form
        $aForm[0] = [
            'form' => [
                'legend' => [
                    'title' => $this->l('Rabo Smart Pay - Settings'),
                    'icon' => 'icon-cog',
                ],
                'input' => [
                    [
                        'type' => 'text',
                        'label' => $this->l('Webhook URL'),
                        'desc' => $this->l('Your webhook URL, please enter this on the Rabo Smart Pay Dashboard.'),
                        'name' => 'ICROK2_WEBHOOK_URL',
                        'required' => true,
                        'disabled' => true,
                        'value' => $this->context->link->getModuleLink($this->name, 'webhook', [], true),
                    ],
                    [
                        'type' => 'textarea',
                        'label' => $this->l('Production - Refresh Token'),
                        'name' => 'ICROK2_REFRESHTOKEN_PRODUCTION',
                        'required' => true,
                    ],
                    [
                        'type' => 'text',
                        'label' => $this->l('Production - Signing Key'),
                        'name' => 'ICROK2_SIGNINGKEY_PRODUCTION',
                        'required' => true,
                    ],
                    [
                        'type' => 'textarea',
                        'label' => $this->l('Sandbox - Refresh Token'),
                        'name' => 'ICROK2_REFRESHTOKEN_SANDBOX',
                        'required' => true,
                    ],
                    [
                        'type' => 'text',
                        'label' => $this->l('Sandbox - Signing Key'),
                        'name' => 'ICROK2_SIGNINGKEY_SANDBOX',
                        'required' => true,
                    ],
                    [
                        'type' => 'radio',
                        'label' => $this->l('Sandbox Mode'),
                        'name' => 'ICROK2_SANDBOX',
                        'required' => false,
                        'is_bool' => true,
                        'values' => [
                            [
                              'id' => 'active_on',
                              'value' => 1,
                              'label' => $this->l('Enabled'),
                            ],
                            [
                              'id' => 'active_off',
                              'value' => 0,
                              'label' => $this->l('Disabled'),
                            ],
                        ],
                    ],
                    [
                        'type' => 'radio',
                        'label' => $this->l('Show payment logo\'s'),
                        'name' => 'ICROK2_LOGOS',
                        'required' => false,
                        'is_bool' => true,
                        'values' => [
                            [
                              'id' => 'active_on',
                              'value' => 1,
                              'label' => $this->l('Enabled'),
                            ],
                            [
                              'id' => 'active_off',
                              'value' => 0,
                              'label' => $this->l('Disabled'),
                            ],
                        ],
                    ],
                    [
                        'type' => 'radio',
                        'label' => $this->l('Enable COF (Card On File)'),
                        'name' => 'ICROK2_COF',
                        'required' => false,
                        'is_bool' => true,
                        'values' => [
                            [
                              'id' => 'active_on',
                              'value' => 1,
                              'label' => $this->l('Enabled'),
                            ],
                            [
                              'id' => 'active_off',
                              'value' => 0,
                              'label' => $this->l('Disabled'),
                            ],
                        ],
                    ],
                ],
                'submit' => [
                    'title' => $this->l('Save'),
                ],
            ],
        ];

        $aForm[1] = [
            'form' => [
                'legend' => [
                    'title' => $this->l('Rabo Smart Pay - Payment Methods'),
                    'icon' => 'icon-sliders',
                ],
                'input' => [
                    [
                        'type' => 'checkbox',
                        'label' => $this->l('Enabled Payment Methods'),
                        'name' => 'ICROK2_PAYMENTMETHODS',
                        'required' => true,
                        'multiple' => true,
                        'values' => [
                            'query' => $this->aPaymentMethods,
                            'id' => 'id',
                            'name' => 'name',
                        ],
                    ],
                ],
                'submit' => [
                    'title' => $this->l('Save'),
                ],
            ],
        ];

        $aFieldValues = [
            'ICROK2_WEBHOOK_URL' => $this->context->link->getModuleLink($this->name, 'webhook', [], true),
            'ICROK2_REFRESHTOKEN_PRODUCTION' => Tools::getValue(
                'ICROK2_REFRESHTOKEN_PRODUCTION',
                Configuration::get('ICROK2_REFRESHTOKEN_PRODUCTION')
            ),
            'ICROK2_SIGNINGKEY_PRODUCTION' => Tools::getValue(
                'ICROK2_SIGNINGKEY_PRODUCTION',
                Configuration::get('ICROK2_SIGNINGKEY_PRODUCTION')
            ),
            'ICROK2_REFRESHTOKEN_SANDBOX' => Tools::getValue(
                'ICROK2_REFRESHTOKEN_SANDBOX',
                Configuration::get('ICROK2_REFRESHTOKEN_SANDBOX')
            ),
            'ICROK2_SIGNINGKEY_SANDBOX' => Tools::getValue(
                'ICROK2_SIGNINGKEY_SANDBOX',
                Configuration::get('ICROK2_SIGNINGKEY_SANDBOX')
            ),
            'ICROK2_SANDBOX' => Tools::getValue(
                'ICROK2_SANDBOX',
                Configuration::get('ICROK2_SANDBOX')
            ),
            'ICROK2_LOGOS' => Tools::getValue(
                'ICROK2_LOGOS',
                Configuration::get('ICROK2_LOGOS')
            ),
            'ICROK2_COF' => Tools::getValue(
                'ICROK2_COF',
                Configuration::get('ICROK2_COF')
            ),
        ];

        $aInputs = [];

        foreach ($this->aPaymentMethods as $aPaymentMethod) {
            $iMethodId = $aPaymentMethod['id'];

            $sPaymentMethodEnabled = Tools::getValue(
                'ICROK2_PAYMENTMETHODS_'.$iMethodId,
                Configuration::get('ICROK2_PAYMENTMETHODS_'.$iMethodId)
            );

            $aFieldValues['ICROK2_PAYMENTMETHODS_'.$iMethodId] = $sPaymentMethodEnabled;

            if ($sPaymentMethodEnabled == 'on') {
                $aInputs[] = [
                    'type' => 'text',
                    'label' => $this->l('Sort Order').' - '.$aPaymentMethod['name'],
                    'name' => 'ICROK2_SORTORDER_'.$iMethodId,
                ];

                $aFieldValues['ICROK2_SORTORDER_'.$iMethodId] = Tools::getValue(
                    'ICROK2_SORTORDER_'.$iMethodId,
                    Configuration::get('ICROK2_SORTORDER_'.$iMethodId)
                );
            }
        }

        // Generate the Form
        $aForm[2] = [
            'form' => [
                'legend' => [
                    'title' => $this->l('Rabo Smart Pay - Sort Settings'),
                    'icon' => 'icon-cog',
                ],
                'input' => $aInputs,
                'submit' => [
                    'title' => $this->l('Save'),
                ],
            ],
        ];

        $sCurrentUrl = $this->context->link->getAdminLink('AdminModules', false);
        $sCurrentUrl .= '&configure='.$this->name;
        $sCurrentUrl .= '&tab_module='.$this->tab;
        $sCurrentUrl .= '&module_name='.$this->name;

        $oFormHelper = new HelperForm();
        $oFormHelper->show_toolbar = false;
        $oFormHelper->id = (int) Tools::getValue('id_carrier');
        $oFormHelper->identifier = $this->identifier;
        $oFormHelper->submit_action = 'submit_'.$this->name;
        $oFormHelper->currentIndex = $sCurrentUrl;
        $oFormHelper->token = Tools::getAdminTokenLite('AdminModules');
        $oFormHelper->tpl_vars = [
            'fields_value' => $aFieldValues,
        ];

        // $this->fields_form = [];

        $this->sHtml .= $oFormHelper->generateForm($aForm);

        return $this->sHtml;
    }

    private function postValidation()
    {
        if (Tools::isSubmit('submit_'.$this->name)) {
            $bSandbox = (bool) Tools::getValue('ICROK2_SANDBOX');
            $sRefreshToken = '';
            $sSigningKey = '';

            if ($bSandbox) {
                $sRefreshToken = Tools::getValue('ICROK2_REFRESHTOKEN_SANDBOX');
                $sSigningKey = Tools::getValue('ICROK2_SIGNINGKEY_SANDBOX');
            } else {
                $sRefreshToken = Tools::getValue('ICROK2_REFRESHTOKEN_PRODUCTION');
                $sSigningKey = Tools::getValue('ICROK2_SIGNINGKEY_PRODUCTION');
            }

            if (!empty($sRefreshToken) && !empty($sSigningKey)) {
                // API to check credentials
                $oOmnikassa = new OmnikassaPayment($sRefreshToken, $sSigningKey);
                $oOmnikassa->setTestmode($bSandbox);

                if ($oOmnikassa->setAccessToken()) {
                    $oOmnikassa->setPaymentBrandList();
                    $aPaymentMethods = $oOmnikassa->getPaymentBrands();

                    if (!empty($aPaymentMethods) && count($aPaymentMethods)) {
                        $sOrderId = 'TEST'.time();
                        $sDescription = 'Order '.$sOrderId;

                        $oOmnikassa->setLanguageCode('nl');
                        $oOmnikassa->setOrder($sOrderId, $sDescription);
                        $oOmnikassa->setOrderAmount(round(rand(100, 1000)), 2);
                        $oOmnikassa->setReturnUrl('https://www.ideal-checkout.nl');

                        $oOmnikassa->getTransaction();

                        if ($oOmnikassa->getTransactionUrl()) {
                            // All good
                        } else {
                            $sMessage = $this->l('Payment could not be started, signing key is probably incorrect.');
                            $sMessage .= $this->l('However the following payment methods should be active: ');

                            foreach ($aPaymentMethods as $sMethod) {
                                $sMessage .= $sMethod.', ';
                            }

                            $sMessage = rtrim($sMessage, ',');
                            $this->aPostErrors[] = $this->l($sMessage);
                        }
                    } else {
                        $this->aPostErrors[] = $this->l('No active payment methods on this account.');
                    }
                } else {
                    $aResponse = json_decode($oOmnikassa->getResponse(), true);
                    $sMessage = $this->l('No "accesstoken" could be generated, please check the configuration.');
                    $sMessage .= $this->l('Keep in mind that there are different tokens for each environment.');

                    if (isset($aResponse['errorCode']) && isset($aResponse['consumerMessage'])) {
                        $sMessage .= 'Response from Smart Pay: Error: ';
                        $sMessage .= $aResponse['errorCode'].' '.$aResponse['consumerMessage'];
                    }

                    $this->aPostErrors[] = $sMessage;
                }
            } else {
                $sMessage = $this->l('Please enter the required data before saving.');
                $this->aPostErrors[] = $sMessage;
            }
        }
    }

    private function postProcess()
    {
        if (Tools::isSubmit('submit_'.$this->name)) {
            Configuration::updateValue(
                'ICROK2_REFRESHTOKEN_PRODUCTION',
                Tools::getValue('ICROK2_REFRESHTOKEN_PRODUCTION')
            );
            Configuration::updateValue(
                'ICROK2_SIGNINGKEY_PRODUCTION',
                Tools::getValue('ICROK2_SIGNINGKEY_PRODUCTION')
            );
            Configuration::updateValue(
                'ICROK2_REFRESHTOKEN_SANDBOX',
                Tools::getValue('ICROK2_REFRESHTOKEN_SANDBOX')
            );
            Configuration::updateValue(
                'ICROK2_SIGNINGKEY_SANDBOX',
                Tools::getValue('ICROK2_SIGNINGKEY_SANDBOX')
            );
            Configuration::updateValue(
                'ICROK2_SANDBOX',
                Tools::getValue('ICROK2_SANDBOX')
            );
            Configuration::updateValue(
                'ICROK2_LOGOS',
                Tools::getValue('ICROK2_LOGOS')
            );
            Configuration::updateValue(
                'ICROK2_COF',
                Tools::getValue('ICROK2_COF')
            );

            foreach ($this->aPaymentMethods as $aPaymentMethod) {
                Configuration::updateValue(
                    'ICROK2_PAYMENTMETHODS_'.$aPaymentMethod['id'],
                    Tools::getValue('ICROK2_PAYMENTMETHODS_'.$aPaymentMethod['id'])
                );

                Configuration::updateValue(
                    'ICROK2_SORTORDER_'.$aPaymentMethod['id'],
                    Tools::getValue('ICROK2_SORTORDER_'.$aPaymentMethod['id'])
                );
            }
        }

        $this->sHtml .= $this->displayConfirmation($this->l('Configuration updated and verified!'));
    }

    public function installOrderState()
    {
        $iPendingState = (int) Configuration::get('ICROK2_OS_PENDING');

        if (!$iPendingState || !Validate::isLoadedObject(new OrderState($iPendingState))) {
            $oOrderState = new OrderState();

            $oOrderState->name = [];
            $aLanguages = Language::getLanguages();

            foreach ($aLanguages as $aLanguage) {
                $sIsoCode = Tools::strtolower($aLanguage['iso_code']);

                switch ($sIsoCode) {
                    case 'fr':
                        $oOrderState->name[$aLanguage['id_lang']] = 'En attente de paiement Smart Pay';
                        break;
                    case 'de':
                        $oOrderState->name[$aLanguage['id_lang']] = 'Warten auf Smart Pay Zahlung';
                        break;
                    case 'nl':
                        $oOrderState->name[$aLanguage['id_lang']] = 'Wachten op Smart Pay betaling';
                        break;
                    default:
                        $oOrderState->name[$aLanguage['id_lang']] = 'Awaiting Smart Pay payment';
                }
            }

            $oOrderState->send_email = false;
            $oOrderState->color = '#4169E1';
            $oOrderState->hidden = false;
            $oOrderState->delivery = false;
            $oOrderState->logable = false;
            $oOrderState->invoice = false;
            $oOrderState->module_name = $this->name;

            if ($oOrderState->id) {
                $oOrderState->update();
            } elseif ($oOrderState->add()) {
                $sSourcePath = _PS_MODULE_DIR_.$this->name.DIRECTORY_SEPARATOR.'views'.DIRECTORY_SEPARATOR;
                $sSourcePath .= 'img'.DIRECTORY_SEPARATOR.'os_icomnikassa.png';

                $sDestinationPath = _PS_ROOT_DIR_.DIRECTORY_SEPARATOR.'img'.DIRECTORY_SEPARATOR;
                $sDestinationPath .= 'os'.DIRECTORY_SEPARATOR.(int) $oOrderState->id.'.gif';
                copy($sSourcePath, $sDestinationPath);
            }

            if (Shop::isFeatureActive()) {
                $aShops = Shop::getShops();

                foreach ($aShops as $aShop) {
                    Configuration::updateValue(
                        'ICROK2_OS_PENDING',
                        (int) $oOrderState->id,
                        false,
                        null,
                        (int) $aShop['id_shop']
                    );
                }
            } else {
                Configuration::updateValue('ICROK2_OS_PENDING', (int) $oOrderState->id);
            }
        }

        return true;
    }

    public function uninstallOrderStates()
    {
        $bResult = true;
        $oCollection = new PrestaShopCollection('OrderState');
        $oCollection->where('module_name', '=', $this->name);
        $aOrderStates = $oCollection->getResults();

        if ($aOrderStates == false) {
            return $bResult;
        }

        foreach ($aOrderStates as $oOrderState) {
            $bResult &= $oOrderState->delete();
        }

        return $bResult;
    }

    public function hookPaymentOptions($aParams)
    {
        if (!$this->active) {
            return;
        }

        if (!$this->checkCurrency($aParams['cart'])) {
            return;
        }

        $aPaymentOptions = [];

        foreach ($this->aPaymentMethods as $aPaymentMethod) {
            $bPaymentMethodEnabled = Configuration::get('ICROK2_PAYMENTMETHODS_'.$aPaymentMethod['id']);

            if ($bPaymentMethodEnabled != 'on') {
                continue;
            }

            $oPaymentOption = new PaymentOption();

            if ((bool) Configuration::get('ICROK2_LOGOS')) {
                $sImageDirectory = '';

                $sViewsDir = _PS_MODULE_DIR_.$this->name.DIRECTORY_SEPARATOR.'views'.DIRECTORY_SEPARATOR;

                if (file_exists($sViewsDir.'img'.DIRECTORY_SEPARATOR.$aPaymentMethod['id'].'.png')) {
                    $sTotalPath = $sViewsDir.'img'.DIRECTORY_SEPARATOR.$aPaymentMethod['id'].'.png';
                    $sImageDirectory = Media::getMediaPath($sTotalPath);
                }

                $oPaymentOption->setLogo($sImageDirectory);
            }

            $aParams = ['payment' => $aPaymentMethod['id']];

            switch ($aPaymentMethod['id']) {
                default:
                    $oPaymentOption->setCallToActionText($aPaymentMethod['name'])
                    ->setAction($this->context->link->getModuleLink($this->name, 'payment', $aParams, true));
                    break;
            }

            $aPaymentOptions[] = $oPaymentOption;
        }

        return $aPaymentOptions;
    }

    public function hookDisplayPaymentReturn($aParams)
    {
        if (!$this->active) {
            return;
        }

        if (Tools::getValue('id_order') && Tools::getValue('success')) {
            $order = new Order(Tools::getValue('id_order'));
            $price = $this->formatPrice($order->getOrdersTotalPaid());
            $isGuest = $this->context->customer->is_guest || !$this->context->customer->id;

            $this->context->smarty->assign([
                'is_guest' => $isGuest,
                'order' => $order,
                'price' => $price,
                'status' => 'success',
            ]);

            return $this->display(__FILE__, 'payment_return_success.tpl');
        }

        return $this->display(__FILE__, 'payment_return_failed.tpl');
    }

    public function checkCurrency($oCart)
    {
        $oCurrency = new Currency($oCart->id_currency);
        $aAllowedCurrencies = $this->getCurrency($oCart->id_currency);

        if (is_array($aAllowedCurrencies)) {
            foreach ($aAllowedCurrencies as $aCurrency) {
                if ($oCurrency->id == $aCurrency['id_currency']) {
                    return true;
                }
            }
        }

        return false;
    }

    private function formatPrice($amount): string
    {
        $currency = \Context::getContext()->currency;

        return Context::getContext()->getCurrentLocale()->formatPrice($amount, $currency->iso_code);
    }
}
