<?php
/**
 * CodeBrain BV       https://www.codebrain.nl.
 *
 * @author      CodeBrain B.V. <info@codebrain.nl>
 * @copyright   Codebrain B.V.
 * @license     https://opensource.org/licenses/bsd-license.php
 *
 * @see        https://www.ideal-checkout.nl/
 */
if (!defined('_PS_VERSION_')) {
    exit;
}
class OmnikassaPayment
{
    private $sRefreshToken = '';
    private $sSigningKey = '';
    private $bTestMode = false;
    private $sCachePath = __DIR__;
    private $sPaymentMethod = false;
    private $sLanguageCode = 'en';

    private $aCustomerBillingData = [];
    private $aCustomerShippingData = [];
    private $aCustomerInformation = [];
    private $aProductData = [];
    private $shoppperRef = '';

    private $cofEnabled = false;

    private $sReturnUrl = '';
    private $sReportUrl = '';

    private $sOrderId = '';
    private $fAmount = false;
    private $sCurrencyCode = 'EUR';
    private $sDescription = '';
    private $sAccessToken = '';
    private $sIssuerId = '';
    private $sResponse = '';

    private $aAllowedMethods = [
        'ideal',
        'maestro',
        'mastercard',
        'paypal',
        'visa',
        'v_pay',
        'bancontact',
        'cards',
    ];

    private $aTransaction = [];
    private $aTransactionResults = [];
    private $aPaymentBrands = [];

    public function __construct($sRefreshToken, $sSigningKey)
    {
        $this->sRefreshToken = $sRefreshToken;
        $this->sSigningKey = $sSigningKey;
    }

    // Should point to directory where cache is strored
    public function setCachePath($sPath = '')
    {
        $this->sCachePath = $sPath;
    }

    public function setLanguageCode($sLanguageCode = false)
    {
        if (is_bool($sLanguageCode)) {
            $this->sLanguageCode = false;

            return true;
        } elseif (is_string($sLanguageCode)) {
            $sLanguageCode = strtoupper(substr($sLanguageCode, 0, 2));

            if (in_array($sLanguageCode, ['NL', 'FR', 'DE', 'ES', 'EN'])) {
                $this->sLanguageCode = $sLanguageCode;

                return true;
            }
        }

        return false;
    }

    public function setPaymentMethod($sPaymentMethod = false)
    {
        if (is_bool($sPaymentMethod)) {
            $this->sPaymentMethod = false;

            return true;
        } elseif (is_string($sPaymentMethod)) {
            $sPaymentMethod = strtolower($sPaymentMethod);

            if (in_array($sPaymentMethod, $this->aAllowedMethods)) {
                $this->sPaymentMethod = $sPaymentMethod;

                return true;
            }

            return false;
        }

        return false;
    }

    public function setTestmode($bEnabled = true)
    {
        return $this->bTestMode = $bEnabled;
    }

    public function setOrder($sOrderId, $sDescription = false)
    {
        $this->sOrderId = $sOrderId;
        $this->sDescription = $sDescription;

        if (empty($this->sDescription)) {
            $this->sDescription = 'Webshop bestelling '.$this->sOrderId;
        }

        return true;
    }

    public function setOrderAmount($fAmount)
    {
        $this->fAmount = $fAmount;

        return true;
    }

    public function setCurrencyCode($sCurrencyCode = false)
    {
        if (is_bool($sCurrencyCode)) {
            $this->sCurrencyCode = false;

            return true;
        } elseif (is_string($sCurrencyCode)) {
            $sCurrencyCode = strtoupper(substr($sCurrencyCode, 0, 3));

            if (in_array($sCurrencyCode, ['EUR'])) {
                $this->sCurrencyCode = $sCurrencyCode;

                return true;
            }
        }

        return false;
    }

    public function setReturnUrl($sReturnUrl = false)
    {
        $this->sReturnUrl = $sReturnUrl;

        return true;
    }

    public function setAccessToken()
    {
        $sCacheFile = false;

        // Used cached access token?
        if (($this->bTestMode == false) && $this->sCachePath) {
            $sStoreHost = md5($_SERVER['SERVER_NAME']);
            $sCacheFile = $this->sCachePath.'token.'.$sStoreHost.'.cache';

            if (!file_exists($sCacheFile)) {
                // Attempt to create cache file
                if (@touch($sCacheFile)) {
                    @chmod($sCacheFile, 0600);
                }
            } elseif (is_readable($sCacheFile) && is_writable($sCacheFile)) {
                // Read data from cache file
                if ($sData = file_get_contents($sCacheFile)) {
                    $aToken = json_decode($sData, true);

                    // Get current time to compare expiration of the access token
                    $sCurrentTimestamp = time();

                    if (isset($aToken['validUntil'])) {
                        // Change the valid until ISO notation to UNIX timestamp
                        $sExpirationTimestamp = strtotime($aToken['validUntil']);

                        if ($sCurrentTimestamp <= $sExpirationTimestamp) {
                            $this->sAccessToken = $aToken['token'];

                            return true;
                        }
                    }
                }
            } else {
                $sCacheFile = false;
            }
        }

        // $sUrl = 'https://betalen.rabobank.nl/omnikassa-api'.($this->bTestMode ? '-sandbox' : '').'/gatekeeper/refresh';
        $sUrl = 'https://api.pay'.($this->bTestMode ? '-sandbox' : '').'.rabobank.nl/omnikassa-api/gatekeeper/refresh';

        $sResponse = idealcheckout_doHttpRequest($sUrl, '', true, 30, false, ['Expect:', 'Authorization: Bearer '.$this->sRefreshToken]);

        if (!empty($sResponse)) {
            $this->sResponse = $sResponse;
            $aToken = json_decode($sResponse, true);

            if (!empty($aToken) && sizeof($aToken)) {
                // Save data in cache?
                if ($sCacheFile) {
                    file_put_contents($sCacheFile, json_encode($aToken));
                }

                if (isset($aToken['token'])) {
                    $this->sAccessToken = $aToken['token'];

                    return true;
                }
            } else {
                idealcheckout_log('Invalid response received from Rabo Smart Pay: '.$sResponse, __FILE__, __LINE__);
            }
        } else {
            idealcheckout_log('No accesstoken could be created, check configuration.', __FILE__, __LINE__);
        }

        return false;
    }

    public function setShopperRef($customerId) {
        $this->shoppperRef = $customerId;
    }

    
    public function enableCof($enabled = false) {
        $this->cofEnabled = $enabled;
    }

    public function setCustomerBillingData($aCustomerBillingData)
    {
        $this->aCustomerBillingData = $aCustomerBillingData;
    }

    public function setCustomerShippingData($aCustomerShippingData)
    {
        $this->aCustomerShippingData = $aCustomerShippingData;
    }

    public function setCustomerInformation($aCustomerInformation)
    {
        $this->aCustomerInformation = $aCustomerInformation;
    }

    public function setProductData($aProductData)
    {
        $this->aProductData = $aProductData;
    }

    public function getTransaction()
    {
        if (empty($this->sSigningKey)) {
            idealcheckout_log('No Refresh token found.', __FILE__, __LINE__);

            $this->aTransaction = ['error' => ['message' => 'No Refresh token found, check module configuration.']];

            return false;
        } elseif (empty($this->sSigningKey)) {
            idealcheckout_log('No Signing Key found.', __FILE__, __LINE__);

            $this->aTransaction = ['error' => ['message' => 'No Signing Key found, check module configuration.']];

            return false;
        } elseif (empty($this->sAccessToken)) {
            idealcheckout_log('No Access token could be generated.', __FILE__, __LINE__);

            $this->aTransaction = ['error' => ['message' => 'No Access token could be generated.']];

            return false;
        } elseif (empty($this->sOrderId)) {
            idealcheckout_log('No order ID found.', __FILE__, __LINE__);

            $this->aTransaction = ['error' => ['message' => 'No order ID found.']];

            return false;
        } elseif (empty($this->fAmount)) {
            idealcheckout_log('No amount found.', __FILE__, __LINE__);

            $this->aTransaction = ['error' => ['message' => 'No amount found.']];

            return false;
        } elseif ($this->fAmount < 0.29) {
            idealcheckout_log('Amount '.number_format($this->fAmount, 2, ',', '').' is to small to process order #'.$this->sOrderId.'.', __FILE__, __LINE__);

            $this->aTransaction = ['error' => ['message' => 'Amount '.number_format($this->fAmount, 2, ',', '').' is to small to process order #'.$this->sOrderId.'.']];

            return false;
        } elseif (empty($this->sReturnUrl)) {
            idealcheckout_log('No return URL found.', __FILE__, __LINE__);

            $this->aTransaction = ['error' => ['message' => 'No return URL found.']];

            return false;
        }

        $aRequest['timestamp'] = date('c', time());
        $aRequest['merchantOrderId'] = $this->sOrderId;
        $aRequest['amount'] = [];
        $aRequest['amount']['currency'] = 'EUR';
        $aRequest['amount']['amount'] = $this->fAmount;
        $aRequest['language'] = $this->sLanguageCode;

        $aRequest['description'] = $this->sDescription;
        $aRequest['merchantReturnURL'] = $this->sReturnUrl;
        $aRequest['skipHppResultPage'] = 'true';

        if ($this->sPaymentMethod) {
            $aRequest['paymentBrand'] = strtoupper($this->sPaymentMethod);
            $aRequest['paymentBrandForce'] = 'FORCE_ONCE';
        }

        if($this->cofEnabled && !empty($this->shoppperRef)) {
            $aRequest['shopperRef'] = (string)$this->shoppperRef;
            $aRequest['paymentBrandMetaData'] = [];

            $aRequest['paymentBrandMetaData']['enableCardOnFile'] = 'true';
        }

        // Optional
        if (sizeof($this->aProductData) && sizeof($this->aCustomerShippingData) && sizeof($this->aCustomerBillingData) && sizeof($this->aCustomerInformation)) {
            $aRequest['orderItems'] = $this->aProductData;
            $aRequest['shippingDetail'] = $this->aCustomerShippingData;
            $aRequest['billingDetail'] = $this->aCustomerBillingData;
            $aRequest['customerInformation'] = $this->aCustomerInformation;
        } elseif (!empty($this->aCustomerInformation)) {
            $aRequest['customerInformation'] = $this->aCustomerInformation;
        }

        // $sApiUrl = 'https://betalen.rabobank.nl/omnikassa-api'.($this->bTestMode ? '-sandbox' : '').'/order/server/api/v2/order';
        $sApiUrl = 'https://api.pay'.($this->bTestMode ? '-sandbox' : '').'.rabobank.nl/omnikassa-api/order/server/api/v2/order';

        $sPostData = json_encode($aRequest);

        $sResponse = idealcheckout_doHttpRequest($sApiUrl, $sPostData, true, 30, false, ['Expect:', 'Authorization: Bearer '.$this->sAccessToken]);

        if (!empty($sResponse)) {
            $this->aTransaction = json_decode($sResponse, true);

            if ($this->aTransaction) {
                if (isset($this->aTransaction['omnikassaOrderId']) && isset($this->aTransaction['redirectUrl'])) {
                    return true;
                } elseif (isset($this->aTransaction['errorCode'], $this->aTransaction['consumerMessage'])) {
                    if (strcasecmp($this->aTransaction['errorCode'], '5005') === 0) {
                        $this->aTransaction = ['error' => ['message' => 'Error: '.$this->aTransaction['errorCode'].'. De betaling kan niet gestart worden omdat de geselecteerde betaalmethode nog niet actief is op het account. Neem contact op met de Rabobank Smart Pay.']];
                    } elseif (strcasecmp($this->aTransaction['errorCode'], '5001') === 0) {
                        $this->aTransaction = ['error' => ['message' => 'Error: '.$this->aTransaction['errorCode'].'. Er is iets mis gegaan bij het signeren van het bericht. Neem contact op met iDEAL Checkout.']];
                    } elseif (strcasecmp($this->aTransaction['errorCode'], '5024') === 0) {
                        $this->aTransaction = ['error' => ['message' => 'Error: '.$this->aTransaction['errorCode'].'. Het verzoek is afgekeurd door de Rabo Smart Pay. De betaalmethode is nog niet actief, of er is een fout opgetreden in het verzoek.']];
                    } else {
                        $this->aTransaction = ['error' => ['message' => 'Unknown response received from Rabo Smart Pay.']];
                    }
                } elseif (isset($this->aTransaction['errorCode'], $this->aTransaction['errorMessage'])) {
                    if (strcasecmp($this->aTransaction['errorCode'], '5001') === 0) {
                        $this->aTransaction = ['error' => ['message' => 'Error: '.$this->aTransaction['errorCode'].'. Het verzoek is afgekeurd door de Rabo Smart Pay.']];
                    } elseif (strcasecmp($this->aTransaction['errorCode'], '5024') === 0) {
                        $this->aTransaction = ['error' => ['message' => 'Error: '.$this->aTransaction['errorCode'].'. Het verzoek is afgekeurd door de Rabo Smart Pay. De betaalmethode is nog niet actief, of er is een fout opgetreden in het verzoek.']];
                    } else {
                        $this->aTransaction = ['error' => ['message' => 'Unknown response received from Rabo Smart Pay.']];
                    }
                }
            } else {
                $this->aTransaction = ['error' => ['message' => 'Cannot decode JSON response.']];
            }
        } else {
            $this->aTransaction = ['error' => ['message' => 'No response received from Rabo Smart Pay.']];
        }

        idealcheckout_log($aRequest, __FILE__, __LINE__);
        idealcheckout_log($sResponse, __FILE__, __LINE__);

        return false;
    }

    public function setPaymentBrandList()
    {
        if ($this->sAccessToken) {
            // Call was made, and the accesstoken has been generated
            // $sApiUrl = 'https://betalen.rabobank.nl/omnikassa-api'.($this->bTestMode ? '-sandbox' : '').'/order/server/api/payment-brands';
            $sApiUrl = 'https://api.pay'.($this->bTestMode ? '-sandbox' : '').'.rabobank.nl/omnikassa-api/order/server/api/payment-brands';


            $sResponse = idealcheckout_doHttpRequest($sApiUrl, '', true, 30, false, ['Expect:', 'Authorization: Bearer '.$this->sAccessToken]);

            if (!empty($sResponse)) {
                $aPaymentBrands = json_decode($sResponse, true);

                if (is_array($aPaymentBrands) && (isset($aPaymentBrands['paymentBrands']) && !empty($aPaymentBrands['paymentBrands']))) {
                    foreach ($aPaymentBrands['paymentBrands'] as $aPaymentMethod) {
                        if (strtolower($aPaymentMethod['status']) == 'active') {
                            $sPaymentName = str_replace('_', '-', strtolower($aPaymentMethod['name']));

                            $this->aPaymentBrands[] = $sPaymentName;
                        }
                    }
                }
            }
        }

        return false;
    }

    public function getResponse()
    {
        if (!empty($this->sResponse)) {
            return $this->sResponse;
        }

        return '';
    }

    public function getPaymentBrands()
    {
        if (!empty($this->aPaymentBrands)) {
            return $this->aPaymentBrands;
        }

        return '';
    }

    public function getTransactionUrl()
    {
        if (!empty($this->aTransaction['redirectUrl'])) {
            return $this->aTransaction['redirectUrl'];
        }

        return false;
    }

    public function getOmnikassaOrderId()
    {
        if (!empty($this->aTransaction['omnikassaOrderId'])) {
            return $this->aTransaction['omnikassaOrderId'];
        }

        return false;
    }

    public function getError()
    {
        if (!empty($this->aTransaction['error']['message'])) {
            return $this->aTransaction['error']['message'];
        }

        return false;
    }

    public function getNotificationData($sAuthToken)
    {
        if (!empty($sAuthToken)) {
            // $sApiUrl = 'https://betalen.rabobank.nl/omnikassa-api'.($this->bTestMode ? '-sandbox' : '').'/order/server/api/v2/events/results/merchant.order.status.changed';
            $sApiUrl = 'https://api.pay'.($this->bTestMode ? '-sandbox' : '').'.rabobank.nl/omnikassa-api/order/server/api/events/results/merchant.order.status.changed';

            $sResponse = idealcheckout_doHttpRequest($sApiUrl, false, true, 30, false, ['Expect:', 'Authorization: Bearer '.$sAuthToken]);

            if (!empty($sResponse)) {
                return $sResponse;
            }
        }

        return '';
    }

    public function isValidRequest($sHashString, $sSignature)
    {
        $sHash = hash_hmac('sha512', $sHashString, base64_decode($this->sSigningKey));

        if (strcmp($sHash, $sSignature) === 0) {
            return true;
        }

        return false;
    }

    public function checkResponse($sOmniKassaStatus)
    {
        if (!empty($sOmniKassaStatus)) {
            if (strcasecmp($sOmniKassaStatus, 'IN_PROGRESS') === 0) {
                return 'PENDING';
            } elseif (strcasecmp($sOmniKassaStatus, 'COMPLETED') === 0) {
                return 'SUCCESS';
            } elseif (strcasecmp($sOmniKassaStatus, 'CANCELLED') === 0) {
                return 'CANCELLED';
            } elseif (strcasecmp($sOmniKassaStatus, 'EXPIRED') === 0) {
                return 'FAILURE';
            }
        }

        return '';
    }

    public function getTransactionFromWebhook($aTransactions)
    {
        foreach ($aTransactions as $aTransaction) {
            if (in_array($aTransaction['status'], ['SUCCESS', 'ACCEPTED'])) {
                return $aTransaction;
            }
        }

        return null;
    }
}
