<?php

namespace Opencart\Catalog\Controller\Extension\Ppp\Payment;

class PppIdeal extends \Opencart\System\Engine\Controller
{
    public function index(): string
    {
        $this->load->language('extension/ppp/payment/ppp');
        // Possible issuerlist here
        $data = [];

        return $this->load->view('extension/ppp/payment/ppp_ideal', $data);
    }

    public function confirm(): void
    {
        $this->load->language('extension/ppp/payment/ppp');
        $this->load->model('checkout/order');

        $return = [];

        if (!$return) {
            $order = $this->model_checkout_order->getOrder($this->session->data['order_id']);

            if ($order) {
                $ppp = new PppPayment($this->config->get('payment_ppp_apikey'));

                $currencyCode = $order['currency_code'];
                $amount = $this->currency->format($order['total'], $currencyCode, false, false);
                $orderId = $order['order_id'];
                $orderNumber = $order['order_id'];
                $description = 'Order '.$orderId;
                $languageCode = $this->language->get('code');

                $paymentMethod = substr($order['payment_method']['code'], 14);
                $paymentCode = substr($order['payment_method']['code'], 10);

                $returnUrl = $this->url->link('extension/ppp/payment/'.$paymentCode.'|callback');
                $notifyUrl = $this->url->link('extension/ppp/payment/'.$paymentCode.'|notify');

                $ppp->setOrder($orderId, $orderNumber, $description);
                $ppp->setOrderAmount($amount);
                $ppp->setCurrencyCode($currencyCode);
                $ppp->setLanguageCode($languageCode);
                $ppp->setPaymentMethod($paymentMethod);
                $ppp->setReturnUrl($returnUrl);
                $ppp->setReportUrl($notifyUrl);
                $ppp->setCustomerIp($order['ip']);

                $ppp->setAccessToken();

                if ($ppp->setTransaction()) {
                    if ($redirectUrl = $ppp->getTransactionUrl()) {
                        $return['redirect'] = $redirectUrl;
                    } else {
                        // No URL received, so there must be something wrong
                        $error = $ppp->getError();

                        $return['error'] = $error['message'];
                    }
                } else {
                    // No URL received, so there must be something wrong
                    $error = $ppp->getError();

                    $return['error'] = $error['message'];
                }
            } else {
                $return['error'] = $this->language->get('error_order');
            }
        }

        $this->response->addHeader('Content-Type: application/json');
        $this->response->setOutput(json_encode($return));
    }

    public function callback(): void
    {
        if (empty($_GET['order_number']) && empty($_GET['order_code']) && empty($_GET['payment_job']) && empty($_GET['signature'])) {
            // Redirect to cart?
            header('Location: '.$this->url->link('checkout/checkout'));
            exit;
        } else {
            $ppp = new PppPayment($this->config->get('payment_ppp_apikey'));
            $ppp->setAccessToken();

            if (!$ppp->isUuid($_GET['payment_job'])) {
                // Redirect to cart?
                header('Location: '.$this->url->link('checkout/checkout'));
                exit;
            }

            if ($validRequest = $ppp->checkReturn($_GET)) {
                $orderId = $_GET['order_number'];
                $paymentJob = $_GET['payment_job'];

                $status = $ppp->getStatus($paymentJob);

                $orderSuccess = $this->config->get('payment_ppp_status_success');
                $orderPending = $this->config->get('payment_ppp_status_pending');
                $orderCancelled = $this->config->get('payment_ppp_status_cancelled');

                $this->load->model('checkout/order');
                $this->load->language('extension/ppp/payment/ppp');

                if ($order = $this->model_checkout_order->getOrder($orderId)) {
                    if (in_array($order['order_status_id'], ['0', $orderPending])) {
                        if (strcasecmp($status, 'SUCCESS') === 0) {
                            $this->model_checkout_order->addHistory($order['order_id'], $orderSuccess, '', true);

                            header('Location:'.$this->url->link('checkout/success'));
                            exit;
                        } elseif (strcasecmp($status, 'PENDING') === 0) {
                            $this->model_checkout_order->addHistory($order['order_id'], $orderPending, '', true);

                            header('Location:'.$this->url->link('checkout/success'));
                            exit;
                        } else {
                            $this->session->data['error'] = $this->language->get('payment_failed');

                            header('Location:'.$this->url->link('checkout/checkout'));
                            exit;
                        }
                    } elseif (strcasecmp($order['status'], $orderSuccess) === 0) {
                        header('Location:'.$this->url->link('checkout/success'));
                        exit;
                    } elseif (strcasecmp($order['status'], $orderCancelled) === 0) {
                        $this->session->data['error'] = $this->language->get('payment_failed');

                        header('Location:'.$this->url->link('checkout/checkout'));
                        exit;
                    } else {
                        header('Location:'.$this->url->link('common/home'));
                        exit;
                    }
                } else {
                    // Cannot load order, error
                    $this->session->data['error'] = $this->language->get('error_order');

                    header('Location:'.$this->url->link('checkout/checkout'));
                    exit;
                }
            } else {
                // Return data invalid, redirect to cart
                header('Location:'.$this->url->link('checkout/checkout'));
                exit;
            }
        }
    }

    public function notify(): void
    {
        $this->load->model('checkout/order');

        $jsonData = @file_get_contents('php://input');
        // $this->log->write($jsonData);

        if (empty($jsonData)) {
            http_response_code(500);
            echo 'Invalid notification call';
        } else {
            $ppp = new PppPayment($this->config->get('payment_ppp_apikey'));
            $ppp->setAccessToken();

            $headers = $this->getHeaders();

            // Set up client
            if ($validRequest = $ppp->checkResponse($jsonData, $headers)) {
                $data = json_decode($jsonData, true);

                if ($ppp->isUuid($data['paymentJob'])) {
                    http_response_code(422);
                    echo 'Invalid payment job';
                    exit;
                }

                $ppp->setReportData($data);

                $status = $ppp->getStatus($data['paymentJob']);

                $orderSuccess = $this->config->get('payment_ppp_status_success');
                $orderPending = $this->config->get('payment_ppp_status_pending');
                $orderCancelled = $this->config->get('payment_ppp_status_cancelled');

                $this->load->model('checkout/order');
                $this->language->load('extension/payment/ppp');

                if ($order = $this->model_checkout_order->getOrder($data['orderId'])) {
                    if (in_array($order['order_status_id'], ['0', $orderPending])) {
                        if (strcasecmp($status, 'SUCCESS') === 0) {
                            $this->model_checkout_order->addOrderHistory($order['order_id'], $orderSuccess, "\n\n".$this->language->get('text_payable')."\n".$this->language->get('text_title'), true);
                        } elseif (strcasecmp($status, 'PENDING') === 0) {
                            $this->model_checkout_order->addOrderHistory($order['order_id'], $orderPending, "\n\n".$this->language->get('text_payable')."\n".$this->language->get('text_title'), true);
                        } else {
                            $this->model_checkout_order->addOrderHistory($order['order_id'], $orderCancelled, "\n\n".$this->language->get('text_payable')."\n".$this->language->get('text_title'), true);
                        }
                    } else {
                        // Nothing to update
                    }
                } else {
                    $this->log->write('RPP: Cannot retrieve order object for ID '.$data['orderId']);

                    // Cant find order
                    http_response_code(500);
                    echo 'Can find order, data incorrect';
                    exit;
                }
            } else {
                $this->log->write('RPP: Could not verify order results. Provided signature is invalid. Contact iDEAL Checkout Support on: +31 522 746060');

                http_response_code(500);
                echo 'Invalid notification call';
                exit;
            }
        }

        // Processed
        http_response_code(200);
        echo 'Notify processed';
        exit;
    }

    private function getHeaders()
    {
        $aHeaders = [];

        foreach ($_SERVER as $key => $value) {
            if (substr($key, 0, 5) == 'HTTP_') {
                $key = str_replace(' ', '-', ucwords(strtolower(str_replace('_', ' ', substr($key, 5)))));

                if ($key == 'X-Signature' || $key == 'Accept-Language') {
                    $key = strtolower($key);
                }

                $aHeaders[$key] = $value;
            } else {
                $aHeaders[$key] = $value;
            }
        }

        return $aHeaders;
    }
}
