<?php

namespace Opencart\Catalog\Controller\Extension\Ppp\Payment;

class PppPayment
{
    private $apiKey;

    private $orderId;
    private $orderNumber;
    private $description;
    private $ipAddress;

    private $amount;
    private $shippingAmount = 0.00;
    private $currencyCode;
    private $allowedCurrencies = ['EUR'];

    private $paymentMethod;
    private $allowedMethods = ['afterpay', 'ideal', 'maestro', 'mastercard', 'paypal', 'visa', 'vpay', 'bancontact', 'sofort'];

    private $languageCode;
    private $allowedLocales = ['en', 'es', 'nl', 'de', 'fr'];

    private $accessToken;
    private $paymentJob;

    private $returnUrl;
    private $reportUrl;

    private $returnData;
    private $reportData;

    private $webhookData;
    private $webhookTransactions;
    private $curlError;
    private $returnHeaders;

    private $shippingData;
    private $billingData;
    private $productData;
    private $customerData;

    public function __construct($apiKey = false)
    {
        $this->apiKey = trim($apiKey);
    }

    public function setApiKey($apiKey)
    {
        $this->apiKey = trim($apiKey);

        return true;
    }

    public function setAccessToken()
    {
        $cacheFile = false;

        // Used cached token?
        $storeCode = md5($_SERVER['SERVER_NAME']);
        $cacheFile = __DIR__.DIRECTORY_SEPARATOR.'token.'.$storeCode.'.cache';

        if (!file_exists($cacheFile)) {
            // Attempt to create cache file
            if (@touch($cacheFile)) {
                @chmod($cacheFile, 0600);
            }
        } elseif (is_readable($cacheFile) && is_writable($cacheFile)) {
            // Read data from cache file
            if ($data = file_get_contents($cacheFile)) {
                $tokenData = json_decode($data, true);

                // Get current time to compare expiration of the access token
                $time = time();

                if (isset($tokenData['expiration'])) {
                    // Change the valid until ISO notation to UNIX timestamp
                    $expirationTime = $tokenData['expiration'];

                    if ($time <= $expirationTime) {
                        $this->accessToken = $tokenData['token'];

                        return true;
                    }
                }
            }
        } else {
            $cacheFile = false;
        }

        $apiUrl = 'https://codebrain-ppp.nl/api/v1/guardian';

        $headers = [
            'Accept: application/json',
            'Authorization: Bearer '.$this->apiKey,
        ];

        list($header, $body) = $this->doRequest($apiUrl, false, $headers);

        if (!empty($body)) {
            if ($tokenData = json_decode($body, true)) {
                // Save data in cache?
                if ($cacheFile) {
                    file_put_contents($cacheFile, json_encode($tokenData));
                }

                $this->accessToken = $tokenData['token'];

                return true;
            }
        }

        return false;
    }

    public function setOrder($orderId, $orderNumber, $description = false)
    {
        $this->orderId = $orderId;
        $this->orderNumber = $orderNumber;
        $this->description = $description;

        if (empty($this->description)) {
            $this->description = 'Webshop bestelling '.$this->orderNumber;
        }

        return true;
    }

    public function setOrderAmount($amount)
    {
        $this->amount = round($amount, 2);

        return true;
    }

    public function setShippingAmount($shippingAmount)
    {
        $this->shippingAmount = $shippingAmount;

        return true;
    }

    public function setCustomerIp($ipAddress)
    {
        $this->ipAddress = $ipAddress;

        return true;
    }

    public function setCurrencyCode($currencyCode = false)
    {
        if (is_bool($currencyCode)) {
            $this->currencyCode = false;

            return true;
        } elseif (is_string($currencyCode)) {
            $currencyCode = strtoupper(substr($currencyCode, 0, 3));

            if (in_array($currencyCode, $this->allowedCurrencies)) {
                $this->currencyCode = $currencyCode;

                return true;
            }
        }

        return false;
    }

    public function setLanguageCode($languageCode = false)
    {
        if (is_bool($languageCode)) {
            $this->languageCode = false;

            return true;
        } elseif (is_string($languageCode)) {
            if (in_array($languageCode, $this->allowedLocales)) {
                // $languageCode = strtoupper(substr($languageCode, 0, 2));
                $this->languageCode = $languageCode;

                return true;
            }
        }

        return false;
    }

    public function setPaymentMethod($paymentMethod = false)
    {
        if (is_bool($paymentMethod)) {
            $this->paymentMethod = false;

            return true;
        } elseif (is_string($paymentMethod)) {
            $paymentMethod = strtolower($paymentMethod);

            if (in_array($paymentMethod, $this->allowedMethods)) {
                $this->paymentMethod = $paymentMethod;

                return true;
            }
        }

        return false;
    }

    public function setReturnUrl($returnUrl = false)
    {
        $this->returnUrl = $returnUrl;

        return true;
    }

    public function setReportUrl($reportUrl = false)
    {
        // Not used in OmniKassa
        $this->reportUrl = $reportUrl;

        return true;
    }

    public function setShippingData($shippingData)
    {
        $this->shippingData = $shippingData;

        return true;
    }

    public function setBillingData($billingData)
    {
        $this->billingData = $billingData;

        return true;
    }

    public function setProductData($productData)
    {
        $this->productData = $productData;

        return true;
    }

    public function setCustomerData($customerData)
    {
        $this->customerData = $customerData;

        return true;
    }

    public function setTransaction()
    {
        if (empty($this->apiKey)) {
            $this->paymentJob = ['error' => ['message' => 'No API Key found, check module configuration.']];

            return false;
        } elseif (empty($this->accessToken)) {
            $this->paymentJob = ['error' => ['message' => 'No Access token could be generated.']];

            return false;
        } elseif (empty($this->orderId)) {
            $this->paymentJob = ['error' => ['message' => 'No order ID found.']];

            return false;
        } elseif (empty($this->amount)) {
            $this->paymentJob = ['error' => ['message' => 'No amount found.']];

            return false;
        } elseif (empty($this->returnUrl)) {
            $this->paymentJob = ['error' => ['message' => 'No return URL found.']];

            return false;
        }

        $request['timestamp'] = time();
        $request['datetime'] = date('c', time());

        $request['currency'] = 'EUR';
        $request['amount'] = $this->amount;
        $request['amount_shipping'] = $this->shippingAmount;

        $request['return_url'] = $this->returnUrl;
        $request['report_url'] = $this->reportUrl;

        $request['payment_method'] = $this->paymentMethod;

        $request['order_id'] = $this->orderId;
        $request['order_number'] = $this->orderNumber;
        $request['language'] = $this->languageCode;
        $request['ip_address'] = $this->ipAddress;

        if (!empty($this->shippingData) && !empty($this->billingData) && !empty($this->productData)) {
            $request['additional_data'] = [];
            $request['additional_data'] = $this->customerData;
            $request['additional_data']['order_lines'] = $this->productData;
            $request['additional_data']['billing_address'] = $this->billingData;
            $request['additional_data']['shipping_address'] = $this->shippingData;
        }

        $postData = json_encode($request);
        $signature = hash_hmac('sha512', $postData, $this->accessToken);

        $apiUrl = 'https://codebrain-ppp.nl/api/v1/transaction';

        $headers = [
            'Content-Type: application/json',
            'Authorization: Bearer '.$this->apiKey,
            'x-signature: '.$signature,
        ];

        list($header, $body) = $this->doRequest($apiUrl, $postData, $headers);

        if (!empty($body)) {
            // Check hash
            $signature = $header['x-signature'];
            $hash = hash_hmac('sha512', $body, $this->accessToken);

            if (hash_equals($signature, $hash)) {
                $this->paymentJob = json_decode($body, true);

                if (is_array($this->paymentJob) && !empty($this->paymentJob)) {
                    if (isset($this->paymentJob['paymentJob']) && isset($this->paymentJob['pay_url'])) {
                        return true;
                    } elseif (isset($this->paymentJob['error'])) {
                        $this->paymentJob = ['error' => ['message' => $this->paymentJob['error']]];
                    }
                } else {
                    $this->paymentJob = ['error' => ['message' => 'Order announcement could not be decoded, something wrong with the data received']];
                }
            } else {
                $this->paymentJob = ['error' => ['message' => 'Response could not be validated, please try again']];
            }
        } else {
            $this->paymentJob = ['error' => ['message' => 'No response received from the Professional Payment Portal.']];
        }

        return false;
    }

    public function getTransactionUrl()
    {
        if (!empty($this->paymentJob['pay_url'])) {
            return $this->paymentJob['pay_url'];
        }

        return false;
    }

    public function getTransactionId()
    {
        if (!empty($this->paymentJob['paymentJob'])) {
            return $this->paymentJob['paymentJob'];
        }

        return false;
    }

    public function getError()
    {
        if (!empty($this->paymentJob['error']['message'])) {
            return $this->paymentJob['error']['message'];
        }

        return false;
    }

    public function checkReturn($data)
    {
        // Required data for RPP
        if (isset($data['order_number']) && isset($data['order_code']) && isset($data['payment_job']) && isset($data['signature'])) {
            $hashString = $data['order_number'].','.$data['order_code'].','.$data['payment_job'];

            $hash = hash_hmac('sha512', $hashString, $this->accessToken);

            if (hash_equals($data['signature'], $hash)) {
                return true;
            }
        }

        return false;
    }

    public function checkResponse($data, $headers)
    {
        // Controleer de signature met de data die is verstuurd
        $hash = hash_hmac('sha512', $data, $this->accessToken);

        $signature = $headers['x-signature'];

        if (hash_equals($signature, $hash)) {
            // Signature matches
            return true;
        }

        return false;
    }

    public function setReturnData($data)
    {
        $this->returnData = $data;

        return true;
    }

    public function setReportData($data)
    {
        $this->reportData = $data;

        return true;
    }

    public function getStatus($paymentJobId)
    {
        $apiUrl = 'https://codebrain-ppp.nl/api/v1/paymentjobs/'.$paymentJobId;

        $headers = [
            'Content-Type: application/json',
            'Authorization: Bearer '.$this->apiKey,
        ];

        list($header, $body) = $this->doRequest($apiUrl, false, $headers);

        if (!empty($body)) {
            // Check hash
            $signature = $header['x-signature'];

            $hash = hash_hmac('sha512', $body, $this->accessToken);

            if (hash_equals($signature, $hash)) {
                $paymentJob = json_decode($body, true);

                if (isset($paymentJob['status'])) {
                    return $paymentJob['status'];
                }
            }
        }

        return false;
    }

    private function doRequest($apiUrl, $postData = false, $headers = [])
    {
        $curl = curl_init();

        curl_setopt($curl, CURLOPT_URL, $apiUrl);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);

        if ($postData != false) {
            curl_setopt($curl, CURLOPT_POST, 1);
            curl_setopt($curl, CURLOPT_POSTFIELDS, $postData);
        }

        curl_setopt($curl, CURLOPT_HTTPHEADER, $headers);
        curl_setopt($curl, CURLOPT_FOLLOWLOCATION, true);
        curl_setopt($curl, CURLOPT_MAXREDIRS, 10);

        curl_setopt($curl, CURLOPT_HEADER, true);

        $body = curl_exec($curl);

        // extract header
        $headerSize = curl_getinfo($curl, CURLINFO_HEADER_SIZE);
        $header = substr($body, 0, $headerSize);
        $header = $this->getHeaders($header);

        // extract body
        $body = substr($body, $headerSize);

        curl_close($curl);

        return [$header, $body];
    }

    private function getHeaders($respHeaders)
    {
        $headers = [];

        $headerText = substr($respHeaders, 0, strpos($respHeaders, "\r\n\r\n"));

        foreach (explode("\r\n", $headerText) as $i => $line) {
            if ($i === 0) {
                $headers['http_code'] = $line;
            } else {
                list($key, $value) = explode(': ', $line);

                $headers[strtolower($key)] = $value;
            }
        }

        return $headers;
    }

    public function isUuid($value)
    {
        if (!is_string($value)) {
            return false;
        }

        return preg_match('/^[\da-f]{8}-[\da-f]{4}-[\da-f]{4}-[\da-f]{4}-[\da-f]{12}$/iD', $value) > 0;
    }
}
