<?php

namespace Opencart\Catalog\Controller\Extension\Rpp\Payment;

class RppAfterpay extends \Opencart\System\Engine\Controller
{
    public function index(): string
    {
        $this->load->language('extension/rpp/payment/rpp');
        // Possible issuerlist here
        $data = [];

        return $this->load->view('extension/rpp/payment/rpp_afterpay', $data);
    }

    public function confirm(): void
    {
        $this->load->language('extension/rpp/payment/rpp');
        $this->load->model('checkout/order');

        $return = [];

        if (!$return) {
            $order = $this->model_checkout_order->getOrder($this->session->data['order_id']);

            if ($order) {
                $rpp = new RppPayment($this->config->get('payment_rpp_apikey'));

                if (strpos($order['payment_method']['code'], '.') !== false) {
                    // Explode payment method
                    $paymentMethod = explode('.', $order['payment_method']['code']);
                    $paymentCode = $paymentMethod[1];

                    // Trim first 4 characters of payment code for the method
                    $paymentMethod = substr($paymentMethod[0], 4);
                } else {
                    $paymentMethod = substr($order['payment_method']['code'], 17);
                    $paymentCode = substr($order['payment_method']['code'], 13);
                }

                $currencyCode = $order['currency_code'];
                $amount = $this->currency->format($order['total'], $currencyCode, false, false);
                $orderId = $order['order_id'];
                $orderNumber = $order['order_id'];
                $description = 'Order '.$orderId;
                $languageCode = $this->language->get('code');

                $returnUrl = $this->url->link('extension/rpp/payment/'.$paymentCode.'|callback');
                $notifyUrl = $this->url->link('extension/rpp/payment/'.$paymentCode.'|notify');

                // Collect billing address
                $billingAddress = [];
                $shippingAddress = [];
                $customerData = [];

                list($streetName, $houseNumber, $addition) = $this->splitAddress($order['shipping_address_1'].' '.$order['shipping_address_2']);

                $shippingAddress['first_name'] = substr($order['shipping_firstname'], 0, 20);
                $shippingAddress['last_name'] = substr($order['shipping_lastname'], 0, 50);
                $shippingAddress['street'] = substr($streetName, 0, 100);
                $shippingAddress['house_number'] = substr($houseNumber, 0, 5);
                $shippingAddress['house_extension'] = $addition;
                $shippingAddress['postcode'] = substr($order['shipping_postcode'], 0, 10);
                $shippingAddress['city'] = substr($order['shipping_city'], 0, 40);
                $shippingAddress['state'] = substr($order['shipping_zone'], 0, 40);
                $shippingAddress['country_code'] = substr($order['shipping_iso_code_2'], 0, 2);

                $shippingAddress['email'] = substr($order['email'], 0, 45);
                $shippingAddress['phone_number'] = $order['telephone'];

                $billingAddress = $shippingAddress;

                // Products
                $this->load->model('catalog/product');
                $products = $this->cart->getProducts();

                $orderlines = [];

                foreach ($products as $k => $v) {
                    $product = [];

                    $productInfo = $this->model_catalog_product->getProduct($v['product_id']);

                    $priceExcl = round($this->currency->format($v['price'], $currencyCode, false, false), 2);
                    $priceVat = round($this->tax->getTax($priceExcl, $v['tax_class_id']), 2);
                    $priceIncl = round($priceExcl + $priceVat, 2);

                    $taxRate = round(($priceVat / $priceExcl) * 100, 0);

                    $product['id'] = $v['product_id'];

                    if (!empty($productInfo['sku'])) {
                        $product['code'] = $productInfo['sku'];
                    } else {
                        $product['code'] = $productInfo['product_id'];
                    }

                    $productName = preg_replace('/[^A-Za-z0-9 -]/', '', $productInfo['name']);

                    $product['name'] = substr($productName, 0, 25);

                    $description = html_entity_decode($productInfo['description']);
                    $description = strip_tags($description);
                    $description = trim($description);

                    $product['description'] = substr($description, 0, 100);

                    $product['quantity'] = intval($v['quantity']);

                    $product['price_incl'] = round($priceIncl, 2);

                    $product['price_excl'] = round($priceExcl, 2);
                    $product['vat_amount'] = round($priceVat, 2);
                    $product['vat'] = (float) $taxRate;

                    $orderlines[] = $product;
                }

                // Shipping
                if ($this->cart->hasShipping()) {
                    $shipping = explode('.', $this->session->data['shipping_method']);

                    if (isset($shipping[0]) && isset($shipping[1]) && isset($this->session->data['shipping_methods'][$shipping[0]]['quote'][$shipping[1]])) {
                        // Shipping information
                        $shippingInfo = $this->session->data['shipping_methods'][$shipping[0]]['quote'][$shipping[1]];

                        $priceExcl = round($this->currency->format($shippingInfo['cost'], $currencyCode, false, false), 2);
                        $priceVat = round($this->tax->getTax($priceExcl, $shippingInfo['tax_class_id']), 2);
                        $priceIncl = round($priceExcl + $priceVat, 2);

                        $taxRate = round(($priceVat / $priceExcl) * 100, 0);

                        $product['id'] = $shippingInfo['code'];
                        $product['code'] = $shippingInfo['code'];

                        $productName = preg_replace('/[^A-Za-z0-9 -]/', '', $shippingInfo['title']);

                        $product['name'] = substr($productName, 0, 25);

                        $description = html_entity_decode($productName);
                        $description = strip_tags($description);
                        $description = trim($description);

                        $product['description'] = substr($description, 0, 100);

                        $product['quantity'] = 1;

                        $product['price_incl'] = round($priceIncl, 2);

                        $product['price_excl'] = round($priceExcl, 2);
                        $product['vat_amount'] = round($priceVat, 2);
                        $product['vat'] = (float) $taxRate;

                        $orderlines[] = $product;
                    }
                }

                $rpp->setOrder($orderId, $orderNumber, $description);
                $rpp->setOrderAmount($amount);
                $rpp->setCurrencyCode($currencyCode);
                $rpp->setLanguageCode($languageCode);
                $rpp->setPaymentMethod($paymentMethod);
                $rpp->setReturnUrl($returnUrl);
                $rpp->setReportUrl($notifyUrl);
                $rpp->setCustomerIp($order['ip']);

                $rpp->setProductData($orderlines);
                $rpp->setShippingData($shippingAddress);
                $rpp->setBillingData($billingAddress);
                $rpp->setCustomerData($customerData);

                $rpp->setAccessToken();

                if ($rpp->setTransaction()) {
                    if ($redirectUrl = $rpp->getTransactionUrl()) {
                        $return['redirect'] = $redirectUrl;
                    } else {
                        // No URL received, so there must be something wrong
                        $error = $rpp->getError();

                        $return['error'] = $error['message'];
                    }
                } else {
                    // No URL received, so there must be something wrong
                    $error = $rpp->getError();

                    $return['error'] = $error['message'];
                }
            } else {
                $return['error'] = $this->language->get('error_order');
            }
        }

        $this->response->addHeader('Content-Type: application/json');
        $this->response->setOutput(json_encode($return));
    }

    public function callback(): void
    {
        if (empty($_GET['order_number']) && empty($_GET['order_code']) && empty($_GET['payment_job']) && empty($_GET['signature'])) {
            // Redirect to cart?
            header('Location: '.$this->url->link('checkout/checkout'));
            exit;
        } else {
            $rpp = new RppPayment($this->config->get('payment_rpp_apikey'));
            $rpp->setAccessToken();

            if ($validRequest = $rpp->checkReturn($_GET)) {
                $orderId = $_GET['order_number'];
                $paymentJob = $_GET['payment_job'];

                $status = $rpp->getStatus($paymentJob);

                $orderSuccess = $this->config->get('payment_rpp_status_success');
                $orderPending = $this->config->get('payment_rpp_status_pending');
                $orderCancelled = $this->config->get('payment_rpp_status_cancelled');

                $this->load->model('checkout/order');
                $this->load->language('extension/rpp/payment/rpp');

                if ($order = $this->model_checkout_order->getOrder($orderId)) {
                    if (in_array($order['order_status_id'], ['0', $orderPending])) {
                        if (strcasecmp($status, 'SUCCESS') === 0) {
                            $this->model_checkout_order->addHistory($order['order_id'], $orderSuccess, '', true);

                            header('Location:'.$this->url->link('checkout/success'));
                            exit;
                        } elseif (strcasecmp($status, 'PENDING') === 0) {
                            $this->model_checkout_order->addHistory($order['order_id'], $orderPending, '', true);

                            header('Location:'.$this->url->link('checkout/success'));
                            exit;
                        } else {
                            $this->session->data['error'] = $this->language->get('payment_failed');

                            header('Location:'.$this->url->link('checkout/checkout'));
                            exit;
                        }
                    } elseif (strcasecmp($order['status'], $orderSuccess) === 0) {
                        header('Location:'.$this->url->link('checkout/success'));
                        exit;
                    } elseif (strcasecmp($order['status'], $orderCancelled) === 0) {
                        $this->session->data['error'] = $this->language->get('payment_failed');

                        header('Location:'.$this->url->link('checkout/checkout'));
                        exit;
                    } else {
                        header('Location:'.$this->url->link('common/home'));
                        exit;
                    }
                } else {
                    // Cannot load order, error
                    $this->session->data['error'] = $this->language->get('error_order');

                    header('Location:'.$this->url->link('checkout/checkout'));
                    exit;
                }
            } else {
                // Return data invalid, redirect to cart
                header('Location:'.$this->url->link('checkout/checkout'));
                exit;
            }
        }
    }

    public function notify(): void
    {
        $this->load->model('checkout/order');

        $jsonData = @file_get_contents('php://input');
        // $this->log->write($jsonData);

        if (empty($jsonData)) {
            http_response_code(500);
            echo 'Invalid notification call';
        } else {
            $rpp = new RppPayment($this->config->get('payment_rpp_apikey'));
            $rpp->setAccessToken();

            $headers = $this->getHeaders();

            // Set up client
            if ($validRequest = $rpp->checkResponse($jsonData, $headers)) {
                $data = json_decode($jsonData, true);

                $rpp->setReportData($data);

                $status = $rpp->getStatus($data['paymentJob']);

                $orderSuccess = $this->config->get('payment_rpp_status_success');
                $orderPending = $this->config->get('payment_rpp_status_pending');
                $orderCancelled = $this->config->get('payment_rpp_status_cancelled');

                $this->load->model('checkout/order');
                $this->language->load('extension/payment/rpp');

                if ($order = $this->model_checkout_order->getOrder($data['orderId'])) {
                    if (in_array($order['order_status_id'], ['0', $orderPending])) {
                        if (strcasecmp($status, 'SUCCESS') === 0) {
                            $this->model_checkout_order->addOrderHistory($order['order_id'], $orderSuccess, "\n\n".$this->language->get('text_payable')."\n".$this->language->get('text_title'), true);
                        } elseif (strcasecmp($status, 'PENDING') === 0) {
                            $this->model_checkout_order->addOrderHistory($order['order_id'], $orderPending, "\n\n".$this->language->get('text_payable')."\n".$this->language->get('text_title'), true);
                        } else {
                            $this->model_checkout_order->addOrderHistory($order['order_id'], $orderCancelled, "\n\n".$this->language->get('text_payable')."\n".$this->language->get('text_title'), true);
                        }
                    } else {
                        // Nothing to update
                    }
                } else {
                    $this->log->write('RPP: Cannot retrieve order object for ID '.$data['orderId']);

                    // Cant find order
                    http_response_code(500);
                    echo 'Can find order, data incorrect';
                    exit;
                }
            } else {
                $this->log->write('RPP: Could not verify order results. Provided signature is invalid. Contact iDEAL Checkout Support on: +31 522 746060');

                http_response_code(500);
                echo 'Invalid notification call';
                exit;
            }
        }

        // Processed
        http_response_code(200);
        echo 'Notify processed';
        exit;
    }

    private function getHeaders()
    {
        $aHeaders = [];

        foreach ($_SERVER as $key => $value) {
            if (substr($key, 0, 5) == 'HTTP_') {
                $key = str_replace(' ', '-', ucwords(strtolower(str_replace('_', ' ', substr($key, 5)))));

                if ($key == 'X-Signature' || $key == 'Accept-Language') {
                    $key = strtolower($key);
                }

                $aHeaders[$key] = $value;
            } else {
                $aHeaders[$key] = $value;
            }
        }

        return $aHeaders;
    }

    private function splitAddress($sAddress)
    {
        // Get everything up to the first number with a regex
        $bHasMatch = preg_match('/^[^0-9]*/', $sAddress, $aMatch);

        // If no matching is possible, return the supplied string as the street
        if (!$bHasMatch) {
            return [$sAddress, '', ''];
        }

        // Remove the street from the sAddress.
        $sAddress = str_replace($aMatch[0], '', $sAddress);
        $sStreetname = trim($aMatch[0]);

        // Nothing left to split, return the streetname alone
        if (strlen($sAddress == 0)) {
            return [$sStreetname, '', ''];
        }

        // Explode sAddress to an array using a multiple explode function
        $aAddress = $this->multiExplodeArray([' ', '-', '|', '&', '/', '_', '\\'], $sAddress);

        // Shift the first element off the array, that is the house number
        $iHousenumber = array_shift($aAddress);

        // If the array is empty now, there is no extension.
        if (count($aAddress) == 0) {
            return [$sStreetname, $iHousenumber, ''];
        }

        // Join together the remaining pieces as the extension.
        $sExtension = substr(implode(' ', $aAddress), 0, 4);

        return [$sStreetname, $iHousenumber, $sExtension];
    }

    private function multiExplodeArray($aDelimiter, $sString)
    {
        $sInput = str_replace($aDelimiter, $aDelimiter[0], $sString);
        $aArray = explode($aDelimiter[0], $sInput);

        return $aArray;
    }
}
