<?php

namespace Opencart\Catalog\Controller\Extension\Rpp\Payment;

class RppCreditcard extends \Opencart\System\Engine\Controller
{
    public function index(): string
    {
        $this->load->language('extension/rpp/payment/rpp');
        // Possible issuerlist here
        $data = [];

        return $this->load->view('extension/rpp/payment/rpp_creditcard', $data);
    }

    public function confirm(): void
    {
        $this->load->language('extension/rpp/payment/rpp');
        $this->load->model('checkout/order');

        $return = [];

        if (!$return) {
            $order = $this->model_checkout_order->getOrder($this->session->data['order_id']);

            if ($order) {
                $rpp = new RppPayment($this->config->get('payment_rpp_apikey'));

                if (strpos($order['payment_method']['code'], '.') !== false) {
                    // Explode payment method
                    $paymentMethod = explode('.', $order['payment_method']['code']);
                    $paymentCode = $paymentMethod[1];

                    // Trim first 4 characters of payment code for the method
                    $paymentMethod = substr($paymentMethod[0], 4);
                } else {
                    $paymentMethod = substr($order['payment_method']['code'], 19);
                    $paymentCode = substr($order['payment_method']['code'], 15);
                }

                $currencyCode = $order['currency_code'];
                $amount = $this->currency->format($order['total'], $currencyCode, false, false);
                $orderId = $order['order_id'];
                $orderNumber = $order['order_id'];
                $description = 'Order '.$orderId;
                $languageCode = $this->language->get('code');

                $returnUrl = $this->url->link('extension/rpp/payment/'.$paymentCode.'|callback');
                $notifyUrl = $this->url->link('extension/rpp/payment/'.$paymentCode.'|notify');

                $rpp->setOrder($orderId, $orderNumber, $description);
                $rpp->setOrderAmount($amount);
                $rpp->setCurrencyCode($currencyCode);
                $rpp->setLanguageCode($languageCode);
                $rpp->setPaymentMethod($paymentMethod);
                $rpp->setReturnUrl($returnUrl);
                $rpp->setReportUrl($notifyUrl);
                $rpp->setCustomerIp($order['ip']);

                $rpp->setAccessToken();

                if ($rpp->setTransaction()) {
                    if ($redirectUrl = $rpp->getTransactionUrl()) {
                        $return['redirect'] = $redirectUrl;
                    } else {
                        // No URL received, so there must be something wrong
                        $error = $rpp->getError();

                        $return['error'] = $error['message'];
                    }
                } else {
                    // No URL received, so there must be something wrong
                    $error = $rpp->getError();

                    $return['error'] = $error['message'];
                }
            } else {
                $return['error'] = $this->language->get('error_order');
            }
        }

        $this->response->addHeader('Content-Type: application/json');
        $this->response->setOutput(json_encode($return));
    }

    public function callback(): void
    {
        if (empty($_GET['order_number']) && empty($_GET['order_code']) && empty($_GET['payment_job']) && empty($_GET['signature'])) {
            // Redirect to cart?
            header('Location: '.$this->url->link('checkout/checkout'));
            exit;
        } else {
            $rpp = new RppPayment($this->config->get('payment_rpp_apikey'));
            $rpp->setAccessToken();

            if ($validRequest = $rpp->checkReturn($_GET)) {
                $orderId = $_GET['order_number'];
                $paymentJob = $_GET['payment_job'];

                $status = $rpp->getStatus($paymentJob);

                $orderSuccess = $this->config->get('payment_rpp_status_success');
                $orderPending = $this->config->get('payment_rpp_status_pending');
                $orderCancelled = $this->config->get('payment_rpp_status_cancelled');

                $this->load->model('checkout/order');
                $this->load->language('extension/rpp/payment/rpp');

                if ($order = $this->model_checkout_order->getOrder($orderId)) {
                    if (in_array($order['order_status_id'], ['0', $orderPending])) {
                        if (strcasecmp($status, 'SUCCESS') === 0) {
                            $this->model_checkout_order->addHistory($order['order_id'], $orderSuccess, '', true);

                            header('Location:'.$this->url->link('checkout/success'));
                            exit;
                        } elseif (strcasecmp($status, 'PENDING') === 0) {
                            $this->model_checkout_order->addHistory($order['order_id'], $orderPending, '', true);

                            header('Location:'.$this->url->link('checkout/success'));
                            exit;
                        } else {
                            $this->session->data['error'] = $this->language->get('payment_failed');

                            header('Location:'.$this->url->link('checkout/checkout'));
                            exit;
                        }
                    } elseif (strcasecmp($order['status'], $orderSuccess) === 0) {
                        header('Location:'.$this->url->link('checkout/success'));
                        exit;
                    } elseif (strcasecmp($order['status'], $orderCancelled) === 0) {
                        $this->session->data['error'] = $this->language->get('payment_failed');

                        header('Location:'.$this->url->link('checkout/checkout'));
                        exit;
                    } else {
                        header('Location:'.$this->url->link('common/home'));
                        exit;
                    }
                } else {
                    // Cannot load order, error
                    $this->session->data['error'] = $this->language->get('error_order');

                    header('Location:'.$this->url->link('checkout/checkout'));
                    exit;
                }
            } else {
                // Return data invalid, redirect to cart
                header('Location:'.$this->url->link('checkout/checkout'));
                exit;
            }
        }
    }

    public function notify(): void
    {
        $this->load->model('checkout/order');

        $jsonData = @file_get_contents('php://input');
        // $this->log->write($jsonData);

        if (empty($jsonData)) {
            http_response_code(500);
            echo 'Invalid notification call';
        } else {
            $rpp = new RppPayment($this->config->get('payment_rpp_apikey'));
            $rpp->setAccessToken();

            $headers = $this->getHeaders();

            // Set up client
            if ($validRequest = $rpp->checkResponse($jsonData, $headers)) {
                $data = json_decode($jsonData, true);

                $rpp->setReportData($data);

                $status = $rpp->getStatus($data['paymentJob']);

                $orderSuccess = $this->config->get('payment_rpp_status_success');
                $orderPending = $this->config->get('payment_rpp_status_pending');
                $orderCancelled = $this->config->get('payment_rpp_status_cancelled');

                $this->load->model('checkout/order');
                $this->language->load('extension/payment/rpp');

                if ($order = $this->model_checkout_order->getOrder($data['orderId'])) {
                    if (in_array($order['order_status_id'], ['0', $orderPending])) {
                        if (strcasecmp($status, 'SUCCESS') === 0) {
                            $this->model_checkout_order->addOrderHistory($order['order_id'], $orderSuccess, "\n\n".$this->language->get('text_payable')."\n".$this->language->get('text_title'), true);
                        } elseif (strcasecmp($status, 'PENDING') === 0) {
                            $this->model_checkout_order->addOrderHistory($order['order_id'], $orderPending, "\n\n".$this->language->get('text_payable')."\n".$this->language->get('text_title'), true);
                        } else {
                            $this->model_checkout_order->addOrderHistory($order['order_id'], $orderCancelled, "\n\n".$this->language->get('text_payable')."\n".$this->language->get('text_title'), true);
                        }
                    } else {
                        // Nothing to update
                    }
                } else {
                    $this->log->write('RPP: Cannot retrieve order object for ID '.$data['orderId']);

                    // Cant find order
                    http_response_code(500);
                    echo 'Can find order, data incorrect';
                    exit;
                }
            } else {
                $this->log->write('RPP: Could not verify order results. Provided signature is invalid. Contact iDEAL Checkout Support on: +31 522 746060');

                http_response_code(500);
                echo 'Invalid notification call';
                exit;
            }
        }

        // Processed
        http_response_code(200);
        echo 'Notify processed';
        exit;
    }

    private function getHeaders()
    {
        $aHeaders = [];

        foreach ($_SERVER as $key => $value) {
            if (substr($key, 0, 5) == 'HTTP_') {
                $key = str_replace(' ', '-', ucwords(strtolower(str_replace('_', ' ', substr($key, 5)))));

                if ($key == 'X-Signature' || $key == 'Accept-Language') {
                    $key = strtolower($key);
                }

                $aHeaders[$key] = $value;
            } else {
                $aHeaders[$key] = $value;
            }
        }

        return $aHeaders;
    }
}
